#!/usr/bin/perl -w
#
# NAME
#   trap.pl
# DESCRIPTION
#   Trap.pl is run by both Hobbit and is called by SEC to send trap messages
#   to Hobbit.
#
# AUTHOR
#   Andy Farrior,  andy.farrior@victoriacollege.edu
#
# NOTES
# if needed, run program with bbcmd to set environment variables
#
# bbcmd --env=/var/hobbit/server/etc/hobbitserver.cfg trap.pl
#
# CHANGES
#    5/12/2006 - Added link to trap.php script that queries the
#       SNMPTT log database on a MySQL server
#    7/16/2005 - switched to using $BB $BBDISP "hobbitdboard test=trap fields="hostname,validtime" "
#    to get current status of all traps.  Had been reading the BB-HOSTS and hobbitd.chk files to 
#    get the current status.  Thanks Henrik.
#

#
# Need to install Text::Autoformat from CPAN

use Getopt::Std;
use Text::Autoformat;

getopts('SCd:s:t:m:D');

# Should we add a link to the trap.php page to query the SNMPTT SQL log database?
# 1 - yes,  0 - no
my $USE_SNMPTTSQL = 0;
my $TRAPHISTORY = $ENV{'BBSERVERWWWURL'} . "/trap.php?last=25";


# lag is the amount of time before Hobbit changes the status to purple
# assuming hobbitlaunch.cfg is set to run trap.pl every 5min, this time
# should catch any validtimes before Hobbit turns them purple.
#
# validtimes comes from the Hobbit checkpoint file hobbitd.chk
#
my $LAG = 7;  # in minutes


# LIFETIME is the time for a trap message to be valid/displayed in Hobbit
# LIFETIME is in minutes, unless you add an "h" (hours), "d" (days)
#  or "w" (weeks) immediately after the number
$LIFETIME = "12h";  # Hobbit status lifetime

$TEST = "trap";

my $USAGE="$0
 -S   - server trap monitor
 -C   - client mode
   -d time stamp      Wed Jun 22 15:03:53 2005
   -s severity status (Normal|INFORMATIONAL|MINOR|WARNING|SEVERE|MAJOR|CRITICAL)
   -t trapped host    FQDN hostname
   -m message         Translated trap message from SNMPTT
";


unless ($opt_S or $opt_C) {
  print "debug: S or C\n" if $opt_D;
  die "$USAGE\n";
}

if ($opt_C) {
    unless ($opt_d and $opt_s and $opt_t and $opt_m) {
      print "debug: C options\n" if $opt_D;
      die "$USAGE\n";
    }
}



if ($opt_S) {
  # check age of traps
  print "check traps\n" if $opt_D;
  &CheckTraps;

} elsif ($opt_C) {
  print "send traps\n" if $opt_D;
  # send a trap to Hobbit
  &SendTrap;
}

exit;


sub SendTrap {
    my $timestamp   = $opt_d;
    my $severity    = $opt_s;
    my $trappedhost = $opt_t;
    my $message     = autoformat $opt_m;
    our $TEST;
    our $LIFETIME;

   my $traphistory = "$TRAPHISTORY&hostname=$trappedhost";
   my $trapurl = "<a href=\\\"$traphistory\\\">Trap History</a>";

   $trappedhost =~ s/\./,/g;

    CASE: {
        if ($severity =~ /^Normal$/i)        { $color="green"; last CASE; }
        if ($severity =~ /^INFORMATIONAL$/i) { $color="green"; last CASE; }
        if ($severity =~ /^WARNING$/i)       { $color="yellow"; last CASE; }
        if ($severity =~ /^MINOR$/i)         { $color="yellow"; last CASE; }
        if ($severity =~ /^SEVERE$/i)        { $color="red"; last CASE; }
        if ($severity =~ /^MAJOR$/i)         { $color="red"; last CASE; }
        if ($severity =~ /^CRITICAL$/i)      { $color="red"; last CASE; }
        $color="clear"; last CASE;
    }

   my $line = "status+$LIFETIME $trappedhost.$TEST $color $timestamp\n$message ";

   if ($USE_SNMPTTSQL) { $line .= "\n$trapurl\n";}

   print "$line\n" if $opt_D;

   &UpdateHobbit ($line);
}


    
sub CheckTraps {
    my $now = time();
    my $displaytime = localtime();
    
    my $BB         = $ENV{'BB'};
    my $BBDISP     = $ENV{'BBDISP'};
    my $TRAPSTATUS = "$BB $BBDISP \"hobbitdboard test=trap fields=hostname,validtime,color\" "; 

    open (TRAPSTATUS,"$TRAPSTATUS|") or die "can't open hobbitdboard : $!\n";
    while (<TRAPSTATUS>) {
        chomp;
        my ($hostname,$validtime,$color) = split (/\|/,$_);

        print "$hostname|$validtime|$color\n" if $opt_D;

        # leave it alone if it's been disabled
        next if ($color eq "blue");

        # If less than $LAG time left until end of valid time on alert, update alert
        $diff = $validtime - $now;

        if ($diff < ($LAG*60) ) {
            $displaytime = localtime();
            $line =  "status+$LIFETIME $hostname.$TEST green $displaytime\n";
            $line .= "No traps to report\n";

            $hostname =~ s/,/\./g;
            my $traphistory = "$TRAPHISTORY&hostname=$hostname";
            my $trapurl = "<a href=\\\"$traphistory\\\">Trap History</a>";
            if ($USE_SNMPTTSQL) { $line .= "\n$trapurl\n";}

            &UpdateHobbit($line);
        }
     }
     close (TRAPSTATUS);

}

sub UpdateHobbit {
   my ($status)   = @_;
   my $BB         = $ENV{'BB'};
   my $BBDISP     = $ENV{'BBDISP'};
   my $BBDISPLAYS = $ENV{'BBDISPLAYS'};

   print "$status\n" if $opt_D;

   if ($BBDISP eq "0.0.0.0") {
       @displays = split (' ',$BBDISPLAYS); # assuming seperated by whitespace
       foreach $display (@displays) {
           my $cmd = "$BB $display \"$status\" ";
           system ("$cmd");
       } 
   } else {
       my $cmd = "$BB $BBDISP \"$status\" ";
       system ("$cmd");
   }
}
    
